<?php
// Debug específico para estrutura CSV
header('Content-Type: text/plain; charset=utf-8');

function sendLog($message, $level = 'info') {
    echo "[" . date('H:i:s') . "] $message\n";
    flush();
}

try {
    sendLog('=== DEBUG DA ESTRUTURA DO CSV ===');
    
    // Procurar arquivo CSV
    $csvFile = null;
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $csvFile = $_FILES['csv_file']['tmp_name'];
        sendLog('📁 Usando arquivo enviado via POST');
    } else {
        $possibleFiles = ['db.csv', '../uploads/dados.csv', './dados.csv'];
        foreach ($possibleFiles as $file) {
            if (file_exists($file)) {
                $csvFile = $file;
                sendLog("📁 Usando arquivo: $file");
                break;
            }
        }
    }
    
    if (!$csvFile) {
        throw new Exception('Arquivo CSV não encontrado');
    }
    
    // Abrir arquivo
    $handle = fopen($csvFile, 'r');
    if (!$handle) {
        throw new Exception('Erro ao abrir arquivo CSV');
    }
    
    sendLog('📊 Analisando primeiras linhas...');
    
    // Ler primeiras linhas brutas
    $rawLines = [];
    for ($i = 0; $i < 10; $i++) {
        $line = fgets($handle);
        if ($line !== false) {
            $rawLines[] = trim($line);
        }
    }
    rewind($handle);
    
    sendLog('📋 LINHAS BRUTAS (primeiras 3):');
    for ($i = 0; $i < min(3, count($rawLines)); $i++) {
        sendLog("Linha " . ($i + 1) . " (tamanho: " . strlen($rawLines[$i]) . "):");
        sendLog(substr($rawLines[$i], 0, 200) . (strlen($rawLines[$i]) > 200 ? '...' : ''));
        sendLog('');
    }
    
    // Testar diferentes delimitadores
    $delimiters = [';', ',', '\t', '|'];
    $bestDelimiter = ';';
    $maxColumns = 0;
    
    sendLog('🔍 TESTANDO DELIMITADORES:');
    foreach ($delimiters as $testDelimiter) {
        $delimiter = ($testDelimiter === '\t') ? "\t" : $testDelimiter;
        $columns = str_getcsv($rawLines[0], $delimiter);
        
        sendLog("Delimitador '$testDelimiter': " . count($columns) . " colunas");
        
        if (count($columns) > $maxColumns) {
            $maxColumns = count($columns);
            $bestDelimiter = $delimiter;
        }
    }
    
    sendLog("✅ Melhor delimitador: '$bestDelimiter' com $maxColumns colunas");
    sendLog('');
    
    // Usar o melhor delimitador
    $delimiter = $bestDelimiter;
    
    // Ler cabeçalho
    $header = fgetcsv($handle, 0, $delimiter);
    sendLog('📋 CABEÇALHO (' . count($header) . ' campos):');
    for ($i = 0; $i < count($header); $i++) {
        sendLog("  $i: " . trim($header[$i]));
    }
    sendLog('');
    
    // Ler primeiras 5 linhas de dados
    sendLog('📊 PRIMEIRAS 5 LINHAS DE DADOS:');
    for ($lineNum = 1; $lineNum <= 5; $lineNum++) {
        $row = fgetcsv($handle, 0, $delimiter);
        if ($row === false) break;
        
        sendLog("Linha $lineNum:");
        sendLog("  - Número de campos: " . count($row));
        sendLog("  - Campos vs Cabeçalho: " . count($row) . " vs " . count($header));
        
        if (count($row) !== count($header)) {
            sendLog("  ❌ INCONSISTÊNCIA DETECTADA!");
            
            // Mostrar diferenças
            if (count($row) > count($header)) {
                sendLog("  - CAMPOS EXTRAS: " . (count($row) - count($header)));
                sendLog("  - Campos extras: " . implode(' | ', array_slice($row, count($header))));
            } else {
                sendLog("  - CAMPOS FALTANDO: " . (count($header) - count($row)));
            }
        } else {
            sendLog("  ✅ Estrutura OK");
        }
        
        // Mostrar primeiros campos para análise
        sendLog("  - Primeiros 5 campos:");
        for ($i = 0; $i < min(5, count($row)); $i++) {
            $value = trim($row[$i]);
            sendLog("    $i: " . substr($value, 0, 50) . (strlen($value) > 50 ? '...' : ''));
        }
        
        // Mostrar últimos campos se houver problema
        if (count($row) !== count($header)) {
            sendLog("  - Últimos 3 campos:");
            $start = max(0, count($row) - 3);
            for ($i = $start; $i < count($row); $i++) {
                $value = trim($row[$i]);
                sendLog("    $i: " . substr($value, 0, 50) . (strlen($value) > 50 ? '...' : ''));
            }
        }
        
        sendLog('');
    }
    
    fclose($handle);
    
    // Análise específica para delimitadores
    sendLog('🔧 ANÁLISE DE DELIMITADORES NO CONTEÚDO:');
    $content = file_get_contents($csvFile);
    
    $delimiter_counts = [
        ';' => substr_count($content, ';'),
        ',' => substr_count($content, ','),
        '\t' => substr_count($content, "\t"),
        '|' => substr_count($content, '|')
    ];
    
    foreach ($delimiter_counts as $delim => $count) {
        sendLog("Caractere '$delim': $count ocorrências");
    }
    
    sendLog('');
    sendLog('🔍 POSSÍVEIS CAUSAS:');
    sendLog('1. Campos com quebras de linha (endereços longos)');
    sendLog('2. Aspas não fechadas corretamente');
    sendLog('3. Caracteres especiais não escapados');
    sendLog('4. Delimitador errado');
    sendLog('5. Coluna vazia no final');
    
    sendLog('');
    sendLog('💡 SUGESTÕES:');
    sendLog('1. Abrir CSV no Excel e salvar novamente');
    sendLog('2. Verificar se há campos com quebras de linha');
    sendLog('3. Usar encoding UTF-8');
    sendLog('4. Remover colunas vazias no final');
    
} catch (Exception $e) {
    sendLog('❌ ERRO: ' . $e->getMessage());
}
?>